import os
import subprocess
import random
import hashlib
import threading
import time
import ttkbootstrap as tb
from ttkbootstrap.constants import *
from tkinter import filedialog, messagebox

# ===========================================================
#                       HASH UTILS
# ===========================================================

def file_md5(path):
    """Tạo MD5 hash cho file."""
    h = hashlib.md5()
    with open(path, "rb") as f:
        for chunk in iter(lambda: f.read(4096), b""):
            h.update(chunk)
    return h.hexdigest()

def remove_duplicate_videos(folder, log):
    """Xóa file trùng lặp dựa trên MD5."""
    log("🔍 Đang kiểm tra video trùng lặp...")

    hashes = {}
    removed = 0

    for f in os.listdir(folder):
        if not f.lower().endswith(".mp4"):
            continue

        full = os.path.join(folder, f)
        h = file_md5(full)

        if h in hashes:
            os.remove(full)
            removed += 1
            log(f"⚠️ Xóa bản trùng: {f}")
        else:
            hashes[h] = f

    log(f"✔ Hoàn tất lọc trùng. Đã xóa {removed} video.\n")


# ===========================================================
#             GHÉP VIDEO VÀ XỬ LÝ CHÍNH
# ===========================================================

def merge_group(files, input_dir, output_dir):
    list_path = "list.txt"
    with open(list_path, "w", encoding="utf-8") as f:
        for vid in files:
            f.write(f"file '{os.path.join(input_dir, vid)}'\n")

    base_name = "_".join([os.path.splitext(f)[0] for f in files]) + ".mp4"
    output_path = os.path.join(output_dir, base_name)

    cmd = [
        "ffmpeg", "-y", "-f", "concat", "-safe", "0",
        "-i", list_path,
        "-c", "copy", output_path
    ]

    subprocess.run(
        cmd,
        stdout=subprocess.DEVNULL,
        stderr=subprocess.STDOUT,
        creationflags=subprocess.CREATE_NO_WINDOW
    )



def process_thread(input_dir, output_dir, group_size, logbox):
    """Chạy quá trình trong thread riêng."""
    def log(t):
        logbox.insert("end", t + "\n")
        logbox.see("end")

    # Bước 1: lọc trùng
    remove_duplicate_videos(input_dir, log)

    while True:
        videos = [f for f in os.listdir(input_dir) if f.lower().endswith(".mp4")]

        if len(videos) < 2:
            log("🎉 Hoàn thành! Không còn video để ghép.")
            messagebox.showinfo("Xong!", "Đã ghép xong tất cả video.")
            break

        # Ghép ngẫu nhiên
        random.shuffle(videos)

        # lấy n video ngẫu nhiên
        group = videos[:group_size]
        log(f"🎬 Đang ghép: {group}")

        merge_group(group, input_dir, output_dir)
        log("✔ Ghép xong → Xóa video gốc")

        for v in group:
            os.remove(os.path.join(input_dir, v))

        log("🔁 Tiếp tục vòng lặp...\n")
        time.sleep(0.5)


# ===========================================================
#                          GUI
# ===========================================================

def select_folder(entry):
    f = filedialog.askdirectory()
    if f:
        entry.delete(0, "end")
        entry.insert(0, f)


def start_process(input_entry, output_entry, spin_group, logbox):
    input_dir = input_entry.get().strip()
    output_dir = output_entry.get().strip()
    group_size = int(spin_group.get())

    if not os.path.exists(input_dir):
        messagebox.showerror("Lỗi", "Thư mục video không hợp lệ!")
        return
    if not output_dir:
        messagebox.showerror("Lỗi", "Bạn chưa chọn thư mục xuất video!")
        return

    logbox.delete(1.0, "end")

    # Tạo thread riêng
    thread = threading.Thread(
        target=process_thread,
        args=(input_dir, output_dir, group_size, logbox),
        daemon=True
    )
    thread.start()


# ===========================================================
#                          MAIN
# ===========================================================

app = tb.Window(themename="cyborg")
app.title("⚡ Tuan Bin Auto Ghep Video PRO")
app.geometry("750x550")

# ---- Thêm icon thương hiệu ----
try:
    app.iconbitmap("tuanbjnnd.ico")   # Đổi thành icon của bạn
except:
    pass

# UI
frame = tb.Frame(app, padding=10)
frame.pack(fill="x")

tb.Label(frame, text="Thư mục chứa video (.mp4):").grid(row=0, column=0, sticky="w")
input_entry = tb.Entry(frame, width=50)
input_entry.grid(row=0, column=1, padx=10)
tb.Button(frame, text="Chọn", bootstyle=PRIMARY, command=lambda: select_folder(input_entry)).grid(row=0, column=2)

tb.Label(frame, text="Thư mục Output:").grid(row=1, column=0, pady=10, sticky="w")
output_entry = tb.Entry(frame, width=50)
output_entry.grid(row=1, column=1, padx=10)
tb.Button(frame, text="Chọn", bootstyle=PRIMARY, command=lambda: select_folder(output_entry)).grid(row=1, column=2)

tb.Label(frame, text="Số video ghép mỗi lần:").grid(row=2, column=0)
spin_group = tb.Spinbox(frame, from_=2, to=10, width=10)
spin_group.set(3)
spin_group.grid(row=2, column=1, sticky="w", pady=5)

tb.Button(app,
          text="🚀TUẤN BIN BẮT ĐẦU GHÉP VIDEO (SIÊU TỐC)",
          bootstyle=SUCCESS,
          command=lambda: start_process(input_entry, output_entry, spin_group, logbox)
          ).pack(pady=15)

tb.Label(app, text="Nhật ký xử lý:").pack(anchor="w", padx=10)
logbox = tb.Text(app, height=18)
logbox.pack(fill="both", expand=True, padx=10, pady=5)

app.mainloop()
